' PIC Controlled Battery Capacity Meter
'D. R. G. Mitchell, October 2007, version 1.0

' Code written using the PIC Simulator IDE by OshonSoft v5.21 (www.oshonsoft.com)
' designed for a PICmicro 16F877A - 40 pin microcontroller running at 4MHz
' Microcontroller programmed using the PICALLW Programmer P16Pro (www.picallw.com)


' *************************set up Analogue to Digital Conversion and the pulldown port for the output voltage
' do this first so that the output voltage is pulled down to ground immediately

ADCON1 = 4  'sets all A ports as digital except A0, A1 and  which is used for measuring the voltage and current
' and sets the Vref to Vdd

' also sets left justification - 6 LSBs on adresl are 0

TRISA.0 = 1  ' configure the adc ports as inputs
TRISA.1 = 1
PORTA = 0

Define ADC_CLOCK = 3
Define ADC_SAMPLEUS = 200

TRISD.0 = 0  'sets port d0 as an output - used for the buzzer circuit
Symbol buzzerport = PORTD.0
PORTD.0 = 0

TRISD.1 = 0  ' sets port d1 as an output - connected to the control fet on the op amp output
PORTD.1 = 1  ' during the start up process it turne control fet on so that the power fet's gate is pulled to ground and is turned off


'********************************* Declare variables

Dim timercounter As Byte  ' the time value
timercounter = 0

Dim seconds As Byte  ' the seconds variable
seconds = 0

Dim terminalvolts As Word  ' the voltage at which to terminate the discharge
terminalvolts = 1050  ' this is equivalent to 10.50 volts

Dim dischargeamps As Byte  ' the dischargerate in amps
dischargeamps = 100  'this is equivalent to 1.00 amps

Dim ampsx100 As Word  ' scaling variable for calculating batter capacity
ampsx100 = 0

Dim capacityval As Word  ' storage variable for calculating battery capacity
capacityval = 0

Dim ampsecs As Byte  ' as above
ampsecs = 0

Dim amperehourfraction As Word  ' the battery capacity fractional part (1Ah=10,000)
amperehourfraction = 0

Dim amperehourfractiondisplay As Word  ' as above by /10 to display only 3 sig figs on the LCD
amperehourfractiondisplay = 0

Dim amperehourinteger As Byte  ' the battery capcity integer part
amperehourinteger = 0

Dim minutes As Byte  ' the minutes variable
minutes = 0

Dim hours As Byte  ' the hours variable
hours = 0

Dim i As Word  ' general counting variable
i = 0

Dim waitinterval As Byte  ' a delay variable for controlling button inputs
waitinterval = 3

Dim redacc As Byte  ' an accumulator to store an average to debounce the red button
redacc = 0

Dim blackacc As Byte  ' an accumulator to store an average to debounce the black button
blackacc = 0

Dim buttonpress As Byte  ' a flag which stores which button has been pressed - even=black, odd=red
buttonpress = 0

Dim secondsflag As Byte  ' a flag which is set when the seconds have advanced
secondsflag = 0

Dim voltage As Word  ' the voltage value read in by the ADC
voltage = 0

Dim wholevolts As Word  ' the integer part of the calculated voltage
wholevolts = 0

Dim vfraction As Word  ' the fractional part of the calculated voltage
vfraction = 0

Dim wholecap As Byte  ' the integerpart of the battery capacity
wholecap = 0

Dim fractioncap As Word  'the fracttional part of the battery capacity
fractioncap = 0

Dim batterycapacity As Word  ' the actual battery capacity
batterycapacity = 0

Dim wholecurrent As Byte  ' the integer part of the discharge current
wholecurrent = 0

Dim fractioncurrent As Byte  ' the fractional part of the discharge current
fractioncurrent = 0

Dim truecorrectedcurrent As Word  ' the actual current (corrected for the fact that 1A=204 DAC units)
truecorrectedcurrent = 0

Dim tempword1 As Word  'a temporary word variable 1
tempword1 = 0

Dim tempword2 As Word  'a temporary word variable 2
tempword2 = 0

Dim adcvoltvalue As Word  ' stored the averaged value of the voltage from the ADC on port A0
adcvoltvalue = 0

Dim adccurrentvalue As Word  ' stores the average value of the current from the ADC on port A1
adccurrentvalue = 0

Dim wholetruecurrent As Byte  ' the integer part of the actual current
wholetruecurrent = 0

Dim fractiontruecurrent As Byte  ' the fractional part of the actual current
fractiontruecurrent = 0

Dim averagedischargeamps As Word
averagedischargeamps = 0

Dim intaveragedischargeamps As Word
intaveragedischargeamps = 0

Dim fractaveragedischargeamps As Word
fractaveragedischargeamps = 0

Dim istore As Byte  ' the value read from eprom - the previously saved current
istore = 0

Dim vtermstore As Word  ' the value read from eprom - the previously saved terminal voltage
vtermstore = 0

Dim minutesflag As Byte  ' a flag which is set when the minutes increment
minutesflag = 0

Dim nowamps As Byte  ' the vlaue of current stored in eprom
nowamps = 0

Dim sumamps As Word  ' the sum of all current values saved in eprom
sumamps = 0

Dim epromcounter As Byte  ' a counter to record the number of writes/positions to eprom
epromcounter = 10  ' the first write goes to address 11 the subroutine increments by one first

Dim epromoverflow As Byte  ' a flag set when the eprom is filled and the write starts at address 8 again
' in these circumstances the number of writes is 248 (256-8)
epromoverflow = 0

Dim j As Byte  ' a general byte counting variable
j = 0

Dim everythreemins As Byte  ' tests the minutes to sample every three minutes using mod 3 - gives 12 hours of recording
everythreemins = 0

' clock interrupt is the threshold value for incrementing seconds. This is a function of the processor speed
' the timer runs at the clock speed /4 so a 4Mhz clock speed results in the timer incrementing 1e6 times per second
' The timer generates an interupt when it overflows ie reaches 256, so the number of interupts per second is
' 1e6/256, ie 3906 (for 4Mhz). When the variable timercounter exceeds this threshold, a second has elapsed - so increment
' the seconds variable
' more generally clockinterupt=clock frequency/(4*256*prescaler value)
' We are also using timer prescaling to reduce the number of interrupts. The prescaler is set 1:16
' so 3906 reduces to 3906/16=244

Dim clockinterrupt As Byte
'NOTE: 1 has been subtracted from 244 below as a cludge to improve accuracy
'The accuracy is determined by how close the xtal is to 4MHz - you may need to tweak this value
clockinterrupt = 243  ' this is for 4MHz operation

Symbol redbuttonport = PORTD.6
Symbol blackbuttonport = PORTD.7


'******************************************* set up LCD
' Port B drives the LCD

TRISB = 0  ' all port B are outputs
PORTB = 0

'Port C configurations
'set the ports as inputs to drive the resistor ladder For op amp control
TRISC = 255
PORTC = 0


' Port D configurations
' D0=output to drive buzzer
' D6-D7 = switch inputs

TRISD.7 = 1
PORTD.7 = 0
TRISD.6 = 1
PORTD.6 = 0

'set up the four bit LCD communication on port B
Define LCD_BITS = 4
Define LCD_DREG = PORTB
Define LCD_DBIT = 4
Define LCD_RSREG = PORTB
Define LCD_RSBIT = 1
Define LCD_EREG = PORTB
Define LCD_EBIT = 3
Define LCD_RWREG = PORTD
Define LCD_RWBIT = 2
Define LCD_COMMANDUS = 2500
Define LCD_DATAUS = 50
Define LCD_INITMS = 100

Lcdinit
Lcdcmdout LcdClear

Lcdout "12 Volt Battery"
Lcdcmdout LcdLine2Home
Lcdout "Capacity Meter"
WaitMs 3000
Lcdcmdout LcdClear


'************************************** set up timer

'at 4mhz the timer increments once every 4 clock cycles. With the prescaler off the interrupt will
' trigger 1e6/256 times per second - which is 3906.25. So the counter variable (word) is checked
' if it is >3906 then increment the seconds variable by 1 - ie one second has elapsed.
OPTION_REG = 3  ' sets the prescaler to 1:16 - timer interrupts occur at 1/16th the above freq

'this also sets the other bits of option_reg to zero - which sets the following bits
't0cs=0 use internal clock
't0se=0 increment on low to high transition - not important here
'psa=0 - assign the prescaler to timer 0

INTCON.T0IF = 0  'clears the timer overflow bit
TMR0 = 0  'sets the tmr0 to zero - also clears the prescaler
INTCON.T0IE = 1  'enable Timer0 interrupts


'************************************** Set up the initial parameters and start the discharge

Gosub showpreviousdata  'Recalls data from any previous analysis
Gosub setinitialvalues  ' Prompts the user to set up the discharge parameters
Gosub setcorrectload  ' Prompts the user to insert the correct wattage load (bulb)
Gosub batterycheck  ' Checks to make sure the battery voltage is above the termination voltage
Gosub turnondischargeamps  ' Initiates the discharge

Enable  ' enable  timer 0 interrupts

'************************************** Main Program Loop
loop:

Gosub checktime  ' Checks the time

If secondsflag > 0 Then  ' the time has been incremented - so check the analague values and update the display
	secondsflag = 0  'clear the flag

	Gosub checkanaloguevolts  ' checks the battery voltage
	Gosub checkanaloguecurrent  ' checks the discharge current
	Gosub calculatecapacity  ' increments the battery capacity (t x I)
	Gosub displayvalues  ' updates teh LCD display
	Gosub checkfortermination  ' checks if the battery voltage is below the termination voltage
Endif

If minutesflag = 1 Then Gosub epromwrite  'minutes flag is set to 1 every time the minutes increase by 1
' so once a minute go to epromwrite routine. This checks to see whether to write the current to eprom -
' which it does every set number of minutes

Gosub checkbuttons

Goto loop

End

setcorrectload:  'prompts to set the correct bulb for the current used
Lcdcmdout LcdClear

'prompts for the correct load according to the discharge current
If dischargeamps = 25 Then Lcdout "Use 12V 5W Bulb"

If dischargeamps = 50 Then Lcdout "Use 12V 10W Bulb"
If dischargeamps = 75 Then Lcdout "Use 12V 10W Bulb"

If dischargeamps = 100 Then Lcdout "Use 12V 20W Bulb"
If dischargeamps = 125 Then Lcdout "Use 12V 20W Bulb"
If dischargeamps = 150 Then Lcdout "Use 12V 20W Bulb"

If dischargeamps >= 175 Then Lcdout "Use 12V 35W Bulb"

Lcdcmdout LcdLine2Home
Lcdout "Black to Proceed"

'loops waiting for the black button to be pressed
buttonpress = 0
setloadloop:
	Gosub checkbuttons
	If buttonpress = 2 Then Goto endsetloadloop
Goto setloadloop

endsetloadloop:
Lcdcmdout LcdClear

Return



showpreviousdata:  'accesses and displays the data from the previous run
buttonpress = 0
j = 0

'test to see if their was a previous data set stored - address 0 is set to 128
Read 0, j
WaitMs 5

If j = 128 Then
	Write 0, 0  ' clear the previous data saved flag
	WaitMs 5

	Read 4, sumamps.LB  'read the average current
	WaitMs 5
	Read 5, amperehourinteger  ' the integer part of the battery capacity
	WaitMs 5
	Read 6, amperehourfractiondisplay.HB  ' the fractional part of the capacity (0-999) high byte part
	WaitMs 5
	Read 7, amperehourfractiondisplay.LB  '' the fractional part of the capacity (0-999) low byte part
	WaitMs 5
	Read 8, hours  ' the discharge time in hours
	WaitMs 5
	Read 9, minutes  'the discharge time in minutes
	WaitMs 5
	Read 10, seconds  ' the discharge time in seconds
	WaitMs 5

	' display the data stored from the last analysis

	Lcdcmdout LcdLine1Pos(10)
	If hours > 9 Then Lcdcmdout LcdLine1Pos(9)
	If hours > 99 Then Lcdcmdout LcdLine1Pos(8)

	Lcdout #hours, ":"

	If minutes < 10 Then
		Lcdout "0", #minutes, ":"
	Else
		Lcdout #minutes, ":"
	Endif

	If seconds < 10 Then
		Lcdout "0", #seconds
	Else	
		Lcdout #seconds, "  "
	Endif

	Lcdcmdout LcdLine2Home

	Lcdout #amperehourinteger, "."
	If amperehourfractiondisplay < 100 Then Lcdout "0"
	If amperehourfractiondisplay < 10 Then Lcdout "0"
	Lcdout #amperehourfractiondisplay, "Ah"

	wholecurrent = sumamps / 100
	fractioncurrent = sumamps - (wholecurrent * 100)

	Lcdcmdout LcdLine2Pos(12)
	Lcdout #wholecurrent, "."
	If fractioncurrent < 10 Then Lcdout "0"
	Lcdout #fractioncurrent, "A"

	buttonpress = 0	
	j = 0
priordataloop:  ' Blinks Prev - Data - Blk:OK' in the top right corner of the display
	Lcdcmdout LcdLine1Pos(1)
	Gosub checkbuttons
	If buttonpress = 2 Then Goto endpriordataloop
	WaitMs 16

	If j = 0 Then Lcdout "Prev  "
	If j = 85 Then Lcdout "Data  "
	If j = 170 Then Lcdout "Blk:OK"
	j = j + 1
Goto priordataloop

endpriordataloop:

'reinitialise all used variables prior to starting main program
j = 0
buttonpress = 0
sumamps = 0
amperehourinteger = 0
amperehourfractiondisplay = 0
hours = 0
minutes = 0
seconds = 0
wholecurrent = 0
fractioncurrent = 0


Endif


Return




epromwrite:  ' simply writes the actual current to eprom every three minutes - so that the average
' current can be determined at the end of the discharge - it may not discharge at a constant rate if a low termination
' voltage is chosen and the battery can not drive sufficient current through the chosen load. NOTE: The eprom data wraps
' around if the discharge time exceeds 12hours (3mins x 245 available bytes in EPROM - the first 10 bytes are used for
' storing previous data). If the discharge current is constant then the final capacity is correct.
' However, if the current falls towards the end of the discharge, the average current is the average of the current
' for the final 12 hours of discharge - which may be slightly lower than actual. This code is designed for small batteries
' where 12h available discharge times are morre than sufficient. If longer discharges are required -
' simply change the line below to suit for example       everythreemins = minutes Mod 4
' this records the current value every 4mins and gives a total discharge time of 4 x 245 = 990 = just over 16 hours etc

everythreemins = minutes Mod 3  'samples every three mins - variable =0 at 3,6,9 mins etc
If everythreemins = 0 Then

	epromcounter = epromcounter + 1
	Write epromcounter, truecorrectedcurrent.LB
	WaitMs 5

	If epromcounter = 255 Then  '- the last eprom address
		epromcounter = 10  ' next write will be to addres 11
		epromoverflow = 1  ' indicates wrap around writing so the number of samples is 244 (256-11)nb 0-10 is 11
	Endif

Endif
minutesflag = 0
Return



checkfortermination:  ' this routine monitors the voltage during discharge and terminates when
' the battery voltage falls below Vterm - note the check for ADC value =682 checks for a voltage condition
' below 9v - the linear regression used to convert ADC values to volts produces silly values below this point

If voltage < terminalvolts Then Goto terminatedischarge
If adcvoltvalue < 682 Then Goto terminatedischarge

Return



terminatedischarge:  ' terminates the discharge
PORTD.1 = 1  ' turn on the control fet and switch off the powerfet
Lcdcmdout LcdClear

'display the battery capacity

Lcdout "Done!"

Lcdcmdout LcdLine1Pos(10)
If hours > 9 Then Lcdcmdout LcdLine1Pos(9)
If hours > 99 Then Lcdcmdout LcdLine1Pos(8)

Lcdout #hours, ":"

If minutes < 10 Then
	Lcdout "0", #minutes, ":"
Else
	Lcdout #minutes, ":"
Endif

If seconds < 10 Then
	Lcdout "0", #seconds
Else	
	Lcdout #seconds, "  "
Endif

Lcdcmdout LcdLine2Home

Lcdout #amperehourinteger, "."
If amperehourfractiondisplay < 100 Then Lcdout "0"
If amperehourfractiondisplay < 10 Then Lcdout "0"
Lcdout #amperehourfractiondisplay, "Ah"

' calculate the average current by recalling the  stored values in eprom and averaging them
' if the epromwrite function above is set to save current values every 3mins, the PIC stores up to 12 hours of data then
' it wraps around - see the description of this in the epromwrite function for more details

If epromoverflow = 1 Then epromcounter = 255  ' the overflow indicates that all bytes 11-255 have been written

sumamps = 0
nowamps = 0

For j = 11 To epromcounter  ' epromcounter records how many current samples have been written to eprom
' loop through the addresses where the current was saved and sum them
	Read j, nowamps
	WaitMs 2
	sumamps = sumamps + nowamps
Next j


If sumamps = 0 Then
	sumamps = dischargeamps  ' in case of early termination the discharge value is used
Else
	sumamps = sumamps / (epromcounter - 10)  ' calculate the average current
Endif

' because real numbers are not supported 1A is = 100 - compute the integer and fractional parts
wholecurrent = 0
fractioncurrent = 0
wholecurrent = sumamps / 100
fractioncurrent = sumamps - (wholecurrent * 100)

' display the average discharge current
Lcdcmdout LcdLine2Pos(12)
Lcdout #wholecurrent, "."
If fractioncurrent < 10 Then Lcdout "0"
Lcdout #fractioncurrent, "A"


' set address 0 to 128 to indicate a valid data set was saved - a flag to indicate a previous analysis reached
' succesful termination a the data was written to eprom
Write 0, 128
WaitMs 5

' save the data to eprom in case it is lost or the battery goes flat
Write 4, sumamps.LB  'Save the average discharge current To address 4
WaitMs 5

' save the capacity as two bytes the integer part 1Ah, 2Ah etc is the first byte
' and the fractional part is saved as a two byte word (amperehourfractiondisplay)
' the takes the value 0-999 nb 345 = X.345Ah

Write 5, amperehourinteger  ' takes values of 0-255Ah
WaitMs 5
Write 6, amperehourfractiondisplay.HB
WaitMs 5
Write 7, amperehourfractiondisplay.LB
WaitMs 5

' discharge current values are stored from address 8 onwards - these have already been accessed \
' by this stage, so overwriting them doesn't matter

Write 8, hours
WaitMs 5
Write 9, minutes
WaitMs 5
Write 10, seconds
WaitMs 5

infiniteloop:  ' an infinite loop - displays the parameters and flashes Done!

WaitMs 1000
Lcdcmdout LcdLine1Pos(1)
Lcdout "     "
WaitMs 500

Lcdcmdout LcdLine1Pos(1)
Lcdout "Done!"
Gosub buzzer
Goto infiniteloop

Return



turnondischargeamps:

' port c switches in various points of the resistor ladder by switching a port from input - high impedance, to output - low impedance
' the resistor chain is longest from +5V to C7 and shortest from +5V to c0
' we're using the resistor ladder circuit in M. Praedko's book, 'Programing and customizing the PICmicro Microcontrollers', p834
' setting a port as an output sets in to ground, and draws current through the resistor chain


If dischargeamps = 25 Then
	TRISC.0 = 0
Endif

If dischargeamps = 50 Then
	TRISC.1 = 0
Endif

If dischargeamps = 75 Then
	TRISC.2 = 0
Endif

If dischargeamps = 100 Then
	TRISC.3 = 0
Endif

If dischargeamps = 125 Then
	TRISC.4 = 0
Endif

If dischargeamps = 150 Then
	TRISC.5 = 0
Endif

If dischargeamps = 175 Then
	TRISC.6 = 0
Endif

If dischargeamps = 200 Then
	TRISC.7 = 0
Endif


PORTD.1 = 0  'turns this port off - so that the control fet on the gate of the power fet is no longer pulled
' to ground but is controlled by the output of the op amp

Return



buzzer:  ' function to make a squeek
buzzerport = 1
WaitMs 200
buzzerport = 0
Return



checkanaloguecurrent:  'checks the analogue current reading on An 3

tempword1 = 0
tempword2 = 0
adccurrentvalue = 0

For i = 1 To 10  ' repeated sampling and then averaging to minimise noise
Adcin 3, tempword1
tempword2 = tempword2 + tempword1
WaitMs 10
Next i
adccurrentvalue = tempword2 / 10

' need to add the 40mA consumed by the PIC circuit to the total current
' this corresponds to (40/1000)*204 = 8
' Note with the LCD backlight on this current increases to 110mA. This extra current (70mA) is NOT factored into
' the capacity calculation and so the backlight MUST should be off during discharge or indicated capacities will be lower than actual

adccurrentvalue = adccurrentvalue + 8

' the sensing resistor is 1 ohm, so a current of 1 amp produces a voltage at the adc of 1 volt, which corresponds to 204 adc units
' so simply diving by 204 gives the current

wholetruecurrent = adccurrentvalue / 204
fractiontruecurrent = adccurrentvalue - (wholetruecurrent * 204)
tempword1 = (fractiontruecurrent * 100) / 204
fractiontruecurrent = tempword1

Return



checkanaloguevolts:  'checck the battery voltage on An 0
'sum the ten set point measurements then take an average (deals with noise)
tempword1 = 0
tempword2 = 0
adcvoltvalue = 0

For i = 1 To 10
	Adcin 0, tempword1
	tempword2 = tempword2 + tempword1
	WaitMs 10
Next i
adcvoltvalue = tempword2 / 10

'the voltage measurement is achieved by a potential divider from the unregulated 12V supply. From
'+ve to ground it consists of a 5kohm pot, one end of which is tied to +ve and the sweeper of which
' is connected to 6.8kohm resistor. The other end of the 6.8kohm resistor is connect to the ADC in port (A0)
' this port is connected to ground via a 5.6k resistor. The pot is adjusted so that when 13.5V is on the supply (the highest likely
' battery voltage, there is 4.95V at the ADC - yielding a value of 1023. A plot of unregulated volts in versus
' ADC values (after the above calibration), yields a straight line. The ADC value at 9.00V corresponds to 682
' The plot of Volts In vs ADC value is also a straight line. To help deal with the small real numbers in the equation
' thereof, the data was modified so that the 9.00V/682 data point corresponded to the origin - ie the voltage
' values had 9.00 subtracted and the ADV values had 682 subtracted. The resulting equation of the straight line
' for this data is then Volts In=0.0132 x ADC value + 0.0116. To do this maths in the PIC everything is multiplied by
' 10,000, then the answer is divided by 100 and 900 added to give the voltage. The reason the voltage is kept at 100x
' its true value is to extract the voltage to two decimal places

voltage = (132 * (adcvoltvalue - 682)) + 116
voltage = (voltage / 100) + 900
wholevolts = voltage / 100
vfraction = voltage - (wholevolts * 100)

Return



checktime:  ' subroutine to check the elapsed time and increment the various time variables

If timercounter > clockinterrupt Then
	timercounter = 0
	seconds = seconds + 1
	secondsflag = 1  ' a flag to show that time has been incremented

	If seconds = 60 Then
		seconds = 0
		minutes = minutes + 1
		minutesflag = 1
	Endif

	If minutes = 60 Then
		minutes = 0
		hours = hours + 1
	Endif


Endif

Return



'*************************************************
checkbuttons:  'an averaging routine to eliminate button bounce
buttonpress = 0
redacc = 0
blackacc = 0

For i = 1 To 10  'poll the button to eliminate the effects of switch bounce
	If blackbuttonport = 0 Then blackacc = blackacc + 1
	If redbuttonport = 0 Then redacc = redacc + 1
Next i

'button press 0=nothing, 1=button 1pressed, 2=button 2, 3=both

If redacc > 5 Then
	buttonpress = buttonpress + 1
	Gosub buzzer
Endif

If blackacc > 5 Then
	buttonpress = buttonpress + 2
	Gosub buzzer
Endif


Return


calculatecapacity:  'function to calculate the battery capacity

' wholetruecurrent is the integer part of the actual current
' fractontruecurrent is the fractional part of the actual current 45=0.45A
' truecorrectedcurrent is the total actual current reverse calculated from the above 2
' this is done rather than use the value directly from the ADC because there is a slight error from this
' 1A = 204ADC units rather than a nice round 200. The wholetruecurrent and fractiontruecurrent have been corrected for this
truecorrectedcurrent = (wholetruecurrent * 100) + fractiontruecurrent
ampsx100 = truecorrectedcurrent * 100  ' multiply the discharge amps (already x100) by another 100
capacityval = capacityval + ampsx100 / 36  '36 comes from 3600s in one hour
ampsecs = capacityval / 100  'calculate the value to add to the Ah display (nb 1Ah = 10000)
capacityval = capacityval - (ampsecs * 100)  ' to avoid rounding any non-integer part of above is kept for the next measurement
amperehourfraction = amperehourfraction + ampsecs

'the above calculates the fractional part of the battery capacity
' if this exceeds 10000 - (1Ah), then it is zeroed and 1 is added to the integer part
If amperehourfraction > 10000 Then
	amperehourfraction = 0
	amperehourinteger = amperehourinteger + 1
Endif

amperehourfractiondisplay = amperehourfraction / 10  ' only used to show the value on the LCD to three sig figs
Return




'*************************************************
displayvalues:  ' output the values to the lcd

Lcdcmdout LcdClear

'display the battery capacity

Lcdout #amperehourinteger, "."
If amperehourfractiondisplay < 100 Then Lcdout "0"
If amperehourfractiondisplay < 10 Then Lcdout "0"
Lcdout #amperehourfractiondisplay, "Ah"

'display the instantaneous current

Lcdcmdout LcdLine1Pos(12)
Lcdout #wholetruecurrent, "."
If fractiontruecurrent < 10 Then Lcdout "0"
Lcdout #fractiontruecurrent, "A"

'display the instantaneous battery voltage

Lcdcmdout LcdLine2Home

	If vfraction < 10 Then  ' ensures that fractional values are displayed correctly
		Lcdout #wholevolts, ".0", #vfraction, "V "
	Else
		Lcdout #wholevolts, ".", #vfraction, "V "
	Endif

		Lcdcmdout LcdLine2Pos(10)
		If hours > 9 Then Lcdcmdout LcdLine2Pos(9)
		If hours > 99 Then Lcdcmdout LcdLine2Pos(8)

		Lcdout #hours, ":"

		If minutes < 10 Then
			Lcdout "0", #minutes, ":"
		Else
			Lcdout #minutes, ":"
		Endif

		If seconds < 10 Then
			Lcdout "0", #seconds
		Else	
			Lcdout #seconds
		Endif

Return



setinitialvalues:  ' reads the previously stored values of terminal voltage and discharge current

' read the previously saved values of discharge current and terminal voltage from eprom
' don't start at zero - PICAL writes 255 to the first data byte when programming - don't know why.
Read 1, istore
Read 2, vtermstore.LB  ' the voltage is a word variable so store it in two bytes
Read 3, vtermstore.HB

'if the values are not zero - ie they have been saved - use them, otherwise use the default values for
' terminalvolts and dischargeamps set in the variable declaration. If the values read from eprom are not zero then use them


If istore > 0 Then dischargeamps = istore
If vtermstore > 0 Then terminalvolts = vtermstore


calibrationloop:

If dischargeamps > 99 Then  ' ie current will be 0.XXA)
	wholecurrent = dischargeamps / 100
	fractioncurrent = dischargeamps - (wholecurrent * 100)
Else  ' current will be X.XXA
	wholecurrent = 0
	fractioncurrent = dischargeamps
Endif

' calculate the integer and fractional parts of the terminal voltage and discharge current
wholevolts = terminalvolts / 100  ' the integer part of the terminal voltage
vfraction = terminalvolts - (wholevolts * 100)  ' the fractional part thereof

Lcdcmdout LcdClear
Lcdout "I:", #wholecurrent, "."

If fractioncurrent = 0 Then
	Lcdout "0", #fractioncurrent
Else
	Lcdout #fractioncurrent
Endif

Lcdout "A V:", #wholevolts, "."
If vfraction = 0 Then
	Lcdout "0", #vfraction
Else
	Lcdout #vfraction
Endif

Lcdout "V"

Lcdcmdout LcdLine2Home
Lcdout "Black:OK Red:Set"

WaitMs 500

Gosub checkbuttons
While buttonpress = 0
Gosub checkbuttons
Wend
' stay in a loop until the OK (green) or set (Red) button is pressed


If buttonpress = 1 Then Gosub setparameters  ' the red button has been pressed
If buttonpress = 2 Then Goto calibrationloopbreak  ' green button pressed so continue
Goto calibrationloop

calibrationloopbreak:  ' the breakoutpoint from the calibrationloop
Lcdcmdout LcdClear


' traps to avoid over run when pushing buttons

Gosub checkbuttons
While buttonpress > 0
	Gosub checkbuttons
Wend
buttonpress = 0
WaitMs 1000
Return



' check the battery voltage and display it. If it is less than the terminal voltage display a warning
' otherwise prompt for green button to go
batterycheck:

batterycheckloop:
Gosub checkanaloguevolts
Lcdcmdout LcdClear
If adcvoltvalue >= 682 Then  ' An ADC value of 682 corresponds to 9.00V - the lowest voltage which will be displayed
	Lcdout "Vbatt:", #wholevolts, "."

	If vfraction < 10 Then
		Lcdout "0", #vfraction, "V"
	Else
		Lcdout #vfraction, "V"
	Endif

	Lcdcmdout LcdLine2Home

	If voltage < terminalvolts Then
		Lcdout "Vbatt  <  Vterm!"
		Gosub buzzer
		WaitMs 100
		Gosub buzzer
		WaitMs 100
		Gosub buzzer
		WaitMs 200
	Endif

	If voltage >= terminalvolts Then Lcdout "Black to Start"


Else
	Lcdout "Vbatt  <  Vterm!"  ' the reason for trapping for both vbatt<vterm and Vbatt <9v is that
	' the voltage calculation equation used here returns 0 for 9v and silly voltages for less than that
	' if one traps just on voltage, once the vbatt goes <9v, it all gets a bit silly
	Gosub buzzer
	WaitMs 100
	Gosub buzzer
	WaitMs 100
	Gosub buzzer
	WaitMs 200

Endif


' need a reasonable delay to avoid lcd flicker ca 300ms
' but as a standalone delay it makes button response slow
' so break it up into smaller delays

If adcvoltvalue < 682 Then  ' tthe battery is at or below 9V to not proceed
	buttonpress = 0
	Goto batterycheckloop
Endif

If voltage < terminalvolts Then  ' the battery voltage is below the terminal voltage do not proceed
	buttonpress = 0
	Goto batterycheckloop
Endif

WaitMs 100
Gosub checkbuttons
If buttonpress = 2 Then Goto batterybreakout
WaitMs 100
Gosub checkbuttons
If buttonpress = 2 Then Goto batterybreakout
WaitMs 100
Gosub checkbuttons
If buttonpress = 2 Then Goto batterybreakout
WaitMs 100
Gosub checkbuttons
If buttonpress = 2 Then Goto batterybreakout
WaitMs 100
Gosub checkbuttons
If buttonpress = 2 Then Goto batterybreakout

Goto batterycheckloop

batterybreakout:
Lcdcmdout LcdClear
Return


' red button pressed so set the discharge current and terminal voltage
' set the discharge current and the terminal voltage
setparameters:

' traps to avoid over run when pushing buttons
buttonpress = 0
Gosub checkbuttons
While buttonpress > 0
	Gosub checkbuttons
Wend
WaitMs 500
buttonpress = 0

	Lcdcmdout LcdClear
	Lcdout "Set Discharge"
	Lcdcmdout LcdLine2Home
	Lcdout "Parameters"
	WaitMs 2000
	Lcdcmdout LcdClear


setidischargeloop:  ' set the discharge current
Lcdcmdout LcdClear
wholecurrent = dischargeamps / 100
fractioncurrent = dischargeamps - (wholecurrent * 100)

'display the current value and give the option to change it with the green button
If fractioncurrent > 10 Then
Lcdout "Idisch:", #wholecurrent, ".", #fractioncurrent, "A"
Else
Lcdout "Idisch:", #wholecurrent, ".0", #fractioncurrent, "A"
Endif
Lcdcmdout LcdLine2Home
Lcdout "Black:OK   Red:+"

WaitMs 500
' delay loop waits for a button push
Gosub checkbuttons
While buttonpress = 0
Gosub checkbuttons
Wend

If buttonpress = 1 Then  ' the red button has been pressed so increment by 0.25 to a maximum of 2.00A, then back to 0.25a
	If dischargeamps < 200 Then  'the upper current is 2.00A - this gives 8 current levels - which can be set by 8 ports
		dischargeamps = dischargeamps + 25
	Else
		dischargeamps = 25
	Endif
Endif

If buttonpress = 2 Then  ' the green button has been pressed and the I disch value accepted
	Goto breakidischargeloop
Endif

Goto setidischargeloop
breakidischargeloop:

Lcdcmdout LcdClear

Gosub checkbuttons
While buttonpress > 0
	Gosub checkbuttons
Wend
buttonpress = 0
WaitMs 500


setvtermloop:  ' set the terminal voltage
Lcdcmdout LcdClear
wholevolts = terminalvolts / 100
vfraction = terminalvolts - (wholevolts * 100)

'display the terminal voltage value and give the option to change it with the green button
If vfraction >= 10 Then
Lcdout "Vterm:", #wholevolts, ".", #vfraction, "V"
Else
Lcdout "Vterm:", #wholevolts, ".0", #vfraction, "V"
Endif
Lcdcmdout LcdLine2Home
Lcdout "Black:OK   Red:+"

WaitMs 500
Gosub checkbuttons  ' waits of a button press
While buttonpress = 0
Gosub checkbuttons
Wend

If buttonpress = 1 Then  ' the red button has been pressed so increment by 100mv from 9000 to 12500mV
terminalvolts = terminalvolts + 10
If terminalvolts > 1250 Then terminalvolts = 900
Endif

If buttonpress = 2 Then  ' the red button has been pressed to use the set value
	Goto breakvtermloop
Endif

Goto setvtermloop
breakvtermloop:
buttonpress = 0

'save these values to eprom for use next time
Write 1, dischargeamps
Write 2, terminalvolts.LB
Write 3, terminalvolts.HB

Return


' the timer interrupt which increments the timercounter variable
On Interrupt
Save System
timercounter = timercounter + 1
INTCON.T0IF = 0  'clears the timer overflow bit

Resume